#ifndef __MNG_FUNCTIONS_H__
#define __MNG_FUNCTIONS_H__

/*
 * All the includes that are needed for code using this module to
 * compile correctly should be #included here.
 */
#include "DX_VOS_BaseTypes.h"
#include "CRYS_RSA_Types.h"
#include "Init_CC.h"
/*
   *  Object %name    : MNG_Functions.h
   *  State           :  %state%
   *  Creation date   :  Aug 4 17:39:24 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief Generic Secure Boot h file (API and structures )
   *
   *  \version 
   *  \author Danb
   *  \remarks Copyright (C) 2004 by Discretix Technologies Ltd.
   *           All Rights reserved
   */


/*-------------------------------
  STRUCTURES
---------------------------------*/
typedef struct
{
   char compName[4];
   char type;
   DxUint32_t major;
   DxUint32_t minor;
   DxUint32_t sub;
   DxUint32_t internal;   

}SEP_Version_t;

/* release version string */
typedef DxInt8_t MwVersionInfo_t[50];


/*------------------------------
    DEFINES
--------------------------------*/

/* the maximum size of the platform wrapped key */
#define DX_MNG_MAX_WRAPPED_PLATFORM_KEY_SIZE_IN_BYTES   60


#define DX_CC_MNG_GPR0_HOST_RESIDENT_ERROR					0x20000
#define DX_CC_MNG_GPR0_HOST_CACHE_ERROR				    	0x40000

/*--------------------------------------
    FUNCTIONS
---------------------------------------*/



/*
*   @brief The functions execution steps:
*            1.	calculate HASH of the OEM RSA public key - SHA2 - 256 on { E || N }
*            2.	calculate ECC of the SHA2-256 digest.
*            3.	burn the SHA256 digest and the ECC to the OTP.
*   
*   @param[in] E[in]- The public exponent.
*   @param[in] N_len[in] - the modulus length.
*   @param[in] N_ptr[in] - the modulus vector.
*
*   Returns:  Status of the operation.
*/
#define DX_MNG_SetOemRsaPubKeyHashOtp(E , N_len , N_ptr) _DX_MNG_SetRsaPubKeyHashOtp(1, \
                                                                                    E, \
                                                                                    N_len, \
                                                                                    N_ptr)

/*
*   @brief The functions execution steps:
*            1.	calculate HASH of the Sjtag RSA public key - SHA2 - 256 on { E || N }
*            2.	calculate ECC of the SHA2-256 digest.
*            3.	burn the SHA256 digest and the ECC to the OTP.
*   
*   @param[in] E[in]- The public exponent.
*   @param[in] N_len[in] - the modulus length.
*   @param[in] N_ptr[in] - the modulus vector.
*
*   Returns:  Status of the operation.
*/                                                                                    
#define DX_MNG_SetSJtagRsaPubKeyHashOtp(E , N_len , N_ptr) _DX_MNG_SetRsaPubKeyHashOtp(0, \
                                                                                      E, \
                                                                                      N_len, \
                                                                                      N_ptr)


/*------------------------------------------------
    FUNCTIONS
--------------------------------------------------*/

/*
*   @brief This function receives the RKEK , calculates the ECC and burns both to the RKEK OTP bits.
*          This function is valid only on chip manufacture or chip device LCS states.
*
*   @param[in] Rkek_len - The RKEK to burn. 
*   @param[in] Rkek_ptr - The RKEK to burn.
*
*   Returns:  Status of the operation.
*/
DxError_t DX_MNG_SetRkekOtp(DxUint16_t    RKEK_len,
                               DxUint8_t     *RKEK_ptr);
         
                               
/*
*   @brief This function retrieves from the user the current monotonic counter that is wrapped by the monotonic counter wrapper key and returns the updated monotonic counter wrapped by the same wrapper key.
*          The CC executes the following: 
*           1. unwraps the MTC wrapper key that is wrapped by the RKEK.
*           2. Unwraps the monotonic counter that is received with wrapper key.
*           3. Increments the monotonic counter.
*           4. Wraps the updated monotonic counter with the wrapper key.
*           5. Returns the updated monotonic counter wrapped by the wrapper key.
*           6. Returns the value of the new counter.
*
*   @param[in] PlatformWrapperKey  - the wrapper key used to unwrapp the monotonic counter . This key is created by the DX_CC_MNG_CreatePlatformWrapperKey API and wrapped by the RKEK.
*   @param[in] PlatformWrapperKeySize - the platform wrapper key size in bytes
*   @param[in/out] WrappedMtc - input : the current monotonic counter wrapped by the platform wrapper key the monotonic counter is defined as a big integer ( Big Endian ).
                                Output: the updated monotonic counter wrapped by the platform key.
*   @param[in] MtcLen - the length on the monotonic counter in bytes.
*   @param[out] Counter_ptr[out] - the updated counter value ( plaintext ).
*   @param[in/out] CounterLen_ptr - input the size of the allocated buffer for the counter if the size is not sufficient the function will return an error.
*
*   Returns:  the actual size of the updated counter + Status of the operation.
*/
DxError_t DX_MNG_IncAndGetMonCounter(
                        DxUint8_t     *PlatformWrapperKey,
                        DxUint32_t    PlatformWrapperKeySize, 
                        DxUint8_t     *WrappedMtc,
                        DxUint32_t    MtcLen,
                        DxUint8_t     *Counter_ptr,
                        DxUint32_t    *CounterLen_ptr);

/*
*   @brief This function retrieves from the user the current monotonic counter that is wrapped by the monotonic counter wrapper key and returns the counter value.
*          The CC executes the following: 
*          1. unwraps the MTC wrapper key that is wrapped by the RKEK.
*          2. Unwraps the monotonic counter that is received with wrapper key.
*          3. Returns the value of the counter.
*   @param[in] PlatformWrapperKey  - the wrapper key used to unwrapp the monotonic counter . This key is created by the DX_CC_MNG_CreatePlatformWrapperKey API and wrapped by the RKEK.
*   @param[in] PlatformWrapperKeySize - the platform wrapper key size in bytes
*   @param[in] WrappedMtc- the current monotonic counter wrapped by the platform wrapper key the monotonic counter is defined as a big integer ( Big Endian ).
*   @param[in] MtcLen - the length on the monotonic counter in bytes.
*   @param[out] Counter_ptr[out] - the counter value ( plaintext ).
*   @param[in/out] CounterLen_ptr[in/out] - input the size of the allocated buffer for the counter if the size is not sufficient the function will return an error.
*                  output: the actual size of the updated counter
*
*   Returns:  Status of the operation.
*/
DxError_t DX_MNG_GetMonCounterVal(
                        DxUint8_t  *PlatformWrapperKey,
                        DxUint32_t PlatformWrapperKeySize,  
                        DxUint8_t  *WrappedMtc,
                        DxUint32_t MtcLen,
                        DxUint8_t  *Counter_ptr,
                        DxUint32_t *CounterLen_ptr);


/*
*   @brief This function creates a monotonic counter with the value 0. the monotonic counter wrapped by the platform key is returned to the user. 
*          The CC executes the following: 
*          1. unwraps the MTC wrapper key that is wrapped by the RKEK.
*          2  creates a monotonic counter with the value 0.
*          3. Wraps the updated monotonic counter with the wrapper key.
*          4. Returns the updated monotonic counter wrapped by the wrapper key.
*          5. Returns the value of the new counter.
*   @param[in] PlatformWrapperKey  - the wrapper key used to unwrapp the monotonic counter . This key is created by the DX_CC_MNG_CreatePlatformWrapperKey API and wrapped by the RKEK.
*   @param[in] PlatformWrapperKeySize - the platform wrapper key size in bytes.
*   @param[out] *MtcLen_ptr[out] - the length on the monotonic that is created counter in bytes.
*   @param[out] UpdatedWrappedMtc[out] - the updated monotonic counter wrapped by the platform key.
*   @param[out] Counter_ptr[out] - the updated counter value ( plaintext ).
*   @param[in/out] CounterLen_ptr[in/out] - input the size of the allocated buffer for the counter if the size is not sufficient the function will return an error.
*              output: the actual size of the updated counter.
*
*   Returns:  Status of the operation.
*/
DxError_t DX_MNG_CreateMonotonicCounter(
                        DxUint8_t  *PlatformWrapperKey,
                        DxUint32_t PlatformWrapperKeySize, 
                        DxUint32_t *MtcLen_ptr,
                        DxUint8_t  *UpdatedWrappedMtc,  
                        DxUint8_t  *Counter_ptr,
                        DxUint32_t *CounterLen_ptr);



/*
*   @brief  The CC executes the following:
*               1) verifying that the SHA256 of { E||N } is correlate with the value stored in the OTP according to the key type.
*               2) Executes the CRYS RSA build operation and returns the UserPubKey object
*   
*   @param[in] keyType - the key type : 1 - OEM , 0 - SJTAG.
*   @param[in] keyIndex - in case of the OEM key, the index of the key
*   @param[in] E[in]- The public exponent.
*   @param[in] N_len[in] - the modulus length.
*   @param[in] N_ptr[in] - the modulus vector.
*   @param[out] UserPubKey_ptr[out] - the RSA public key object.
*
*   Returns:  Status of the operation.
*/
DxError_t DX_MNG_RSA_BuildPlatformPubKey(DxUint32_t             keyType,
                                         DxUint32_t             keyIndex, 
                                         CRYS_RSAUserPubKey_t*  UserPubKey_ptr,
                                         DxUint8_t*             E_ptr,					
                                         DxUint32_t             E_len,
                                 		     DxUint32_t             N_len,
                                         DxUint8_t*             N_ptr);


/*
*   @brief The functions execution steps:
*            1.	calculate HASH of the keyType(OEM | Sjtag) RSA public key - SHA2 - 256 on { E || N }
*            2.	calculate ECC of the SHA2-256 digest.
*            3.	burn the SHA256 digest and the ECC to the OTP.
*   
*   @param[in] keyType - the key type : 1 - OEM , 0 - SJTAG.
*   @param[in] E[in]- The public exponent.
*   @param[in] N_len[in] - the modulus length.
*   @param[in] N_ptr[in] - the modulus vector.
*
*   Returns:  Status of the operation.
*/
DxError_t _DX_MNG_SetRsaPubKeyHashOtp(
                             DxUint32_t    keyType, 
                             DxUint32_t    E,
							               DxUint32_t    N_len,
                             DxUint8_t     *N_ptr);


/*
*   @brief This function creates the AES sub key used for wrapping the CA platform keys and the state file.
*   
*   @param[out] WrappedAesSubKey_ptr - The sub key that was created by the SeP.
*   @param[in/out] SubKeyLen_ptr - On the input: the size of the WrappedAesSubKey buffer allocated for the key.
*                                  On the output: the actual size of the wrapped key - if the size of the buffer 
*                                  allocated is not sufficient the function will return an error. 
*
*   Returns:  Status of the operation.
*/
DxError_t DX_MNG_CreatePlatformWrapperKey (
                   DxUint32_t    *PlatformWrapperKeySize_ptr,        
                   DxUint8_t     *PlatformWrapperKey);

/*
*   @brief This function updates the LCS 
*          
*
*   @param[in] NewLcs - New LCS value. 
*
*   Returns:  Status of the operation.
*/
DxError_t DX_MNG_ChangeLCS(DxUint8_t    NewLcs);

/*
*   @brief This function reads the LCS from SEP
*          
*
*   @param[out] LCS_ptr - The current life cycle state.
*
*   Returns:  Status of the operation.
*/
DxError_t DX_MNG_GetLCS(DxUint32_t    *LCS_ptr);

/*
*   @brief This function Generate random vector up to 4K
*          
*
*   @param[in] RNDSize - The size of the generated vector. 
*   @param[out] output_ptr - The buffer for the vector.
*
*   Returns:  Status of the operation.
*/
DxError_t DX_MNG_GenerateDRNGVector(DxUint32_t RNDSize, DxUint8_t* output_ptr);


/*
*   @brief This function set the mode of the API's. If the mode is blocking, then the API will be blocked until it aquires 
*          access to SEP, and receives a response to SEP. If non-blocking - then if the SEP is currently busy with another transaction, an API will fail with appropriate error
*          and will not block
*          
*
*   @param[in] blockingMode - 1 - blocking mode, 0 - non-blocking mode. 
*
*   Returns:  Status of the operation.
*/
DxError_t DX_MNG_SetAPIBlockingMode(DxUint32_t blockingMode);

/*
*   @brief This function Gets the ROM version
*          
*
*   @param[out] version - buffer to hold the version. 
*
*   Returns:  Status of the operation.
*/
DxError_t DX_MNG_GetROMVersion(SEP_Version_t *version);

/*
*   @brief This function Gets the ROM version
*          
*
*   @param[out] version - buffer to hold the version. 
*
*   Returns:  Status of the operation.
*/
DxError_t DX_MNG_GetRAMVersion(SEP_Version_t *version);


/*
*   @brief This function returns the release version
*
*   @param[in] releaseVersion_ptr  - pointer to the string where the version is set
*
*   Returns:  Status of the operation.
*/
DxError_t DX_MNG_GetMWPackageVersion(MwVersionInfo_t  version);

/*
*   @brief This function clear the warm boot idnetification
*          
*
*   Returns:  Status of the operation.
*/
DxError_t DX_MNG_ClearWarmBootIndication(void);

/*
*   @brief This function reset the internal random seed.
*          
*
*   @param[in] rndSampleCnt - The counter for the random sample. 
*
*   Returns:  Status of the operation.
*/
DxError_t DX_CC_MNG_resetSeed(DxUint32_t rndSampleCnt);


/*
*   @brief This function returns the state of the SEP.
*          
*
*   @param[out] CCResult_ptr - buffer to the state of the SEP. 
*
*   Returns:  In case of fatal error the error code else .
*/
#define DX_MNG_GetSepState DX_CC_StartExt


/*
*   @brief This function notify the SEP that the cache was reloaded and it can start the 
*          cold warm boot procedure
*		   Since the SEP is stuck in endless loop till it get the notification,
*          the notification is passed via register and not shared area message.
*   Returns:  Status of the operation.
*/
DxError_t DX_MNG_RecoverFromIcacheViolation( void );


#define DX_IMAGE_VERIFY_SUCCEED							0x0
#define DX_IMAGE_VERIFY_RESIDENT_IMAGE_FAIL             0x1
#define DX_IMAGE_VERIFY_CACHE_IMAGE_FAIL                0x2
#define DX_IMAGE_VERIFY_RESIDENT_AND_OVERLAY_IMAGE_FAIL (DX_IMAGE_VERIFY_RESIDENT_IMAGE_FAIL|DX_IMAGE_VERIFY_OVERLAY_IMAGE_FAIL)
/*
*   @brief This function reads the images verification status from SEP
*          
*
*   @param[out] Status_ptr - The latest verification status (only after cold boot).
*
*   Returns:  Status of the operation.
*/
DxError_t DX_CC_MNG_IsImageVerifySucceed(DxUint32_t    *Status_ptr);


/*
*   @brief This function updates the ext interrupt status register address 
*          
*
*   @param[in] inRegAddr_ptr - address of the input status register. 
*   @param[in] outRegAddr_ptr - address of the output status register. 
*   The register must be accessiable to the SEP.In case of 0x0 teh register address is not updated
*   Returns:  Status of the operation.
*/
DxError_t DX_MNG_SetExtIntStatusRegsAddr(DxUint32_t InStatusAddr,DxUint32_t OutStatusAddr);

/*
*   @brief This function Update the shared area base address in the SEP
*          
*
*   @param[in] newSharedAreaAddress - The new address of the shared area.
*
*   Returns:  0 in case of success else -1.
*/
DxError_t DX_CC_MNG_UpdateSharedAreaAddress(DxUint32_t  newSharedAreaAddress);


/*
*   @brief This enum defines the different cache counters
*          
*/
typedef enum
{
	DX_CC_MNG_COUNTER_TYPE_TOTAL_HIT = 0,
	DX_CC_MNG_COUNTER_TYPE_TOTAL_OPERATIONS = 1,
	DX_CC_MNG_COUNTER_TYPE_TOTAL_FLUSH = 2,
	DX_CC_MNG_COUNTER_TYPE_RD_HIT = 3,
	DX_CC_MNG_COUNTER_TYPE_RD_OPERATIONS = 4,
	DX_CC_MNG_COUNTER_TYPE_RD_FLUSH = 5,
	DX_CC_MNG_COUNTER_TYPE_WR_HIT = 6,
	DX_CC_MNG_COUNTER_TYPE_WR_OPERATIONS = 7,
	DX_CC_MNG_COUNTER_TYPE_WR_FLUSH = 8,
	DX_CC_MNG_COUNTER_TYPE_MAX_CONIG_TYPE = 0x7FFFFFFF
}DX_CC_MNG_CacheCounterConfigType_t;

/*
*   @brief This enum defines the different cache type
*          
*/
typedef enum
{
	DX_CC_MNG_I_CACHE_COUNTER_TYPE = 0,
	DX_CC_MNG_D_CACHE_COUNTER_TYPE = 1,
	DX_CC_MNG_CACHE_COUNTER_TYPE_MAX_TYPE = 0x7FFFFFFF
}DX_CC_MNG_CacheCounterType_t;



/*
*   @brief This function is used for reseting and configuring the 3 cache counters. 
*   Each counter can be one of the 9 options defined in the enumerator DX_CC_MNG_CacheCounterType_t.       
*
*   @param[in] cfgCnt1 - cache counter configuration for counter1. 
*   @param[in] cfgCnt2 - cache counter configuration for counter2. 
*   @param[in] cfgCnt3 - cache counter configuration for counter3. 
*   Returns:  Status of the operation.
*/

DxError_t DX_CC_MNG_ConfigureAndResetCacheCounters(DX_CC_MNG_CacheCounterType_t       cacheType,
                                                   DX_CC_MNG_CacheCounterConfigType_t cfgCnt1,
                                                   DX_CC_MNG_CacheCounterConfigType_t cfgCnt2,
                                                   DX_CC_MNG_CacheCounterConfigType_t cfgCnt3);


/*
*   @brief This function is used for reading the 3 cache counters. 
*   The read is not reseting the counters so counter can be read few times        
*   during the tests flow without reconfiguration.
*
*
*   @param[in] cnt1Val - buffer for counter1. 
*   @param[in] cnt2Val - buffer for counter2. 
*   @param[in] cnt3Val - buffer for counter3. 
*   Returns:  Status of the operation.
*/

DxError_t DX_CC_MNG_GetCacheCounters(DX_CC_MNG_CacheCounterType_t       cacheType,
                                     DxUint32_t                         *cnt1Val,
                                     DxUint32_t                         *cnt2Val,
                                     DxUint32_t                         *cnt3Val);

 

#endif



